/***
 * Copyright(C) 2009 Naoki Inaba and Hirofumi Fujiwara
 * Puzzle Project Room, Time Intermedia Corporation <puzzle@timedia.co.jp>
 * All rights reserved.
 * We make no gurantees that this code if fit for any purpose.
 * Visit http://puzzle.gr.jp/ for more information.
 ***/
package org.example.sudoku;

import jp.gr.puzzle.npgenerator.v1.Calc;

class Pattern {

    private boolean[][] pattern = new boolean[9][9];
    
    final   static  int[]   min_hints = {28,24,20};
    final   static  int[]   max_hints = {36,32,27};
    
    public boolean[][] generateHint(int lev) {
        int     minh = min_hints[lev];
        int     maxh = max_hints[lev];
        int     hw = maxh - minh + 1;
        do {
            int count = (Calc.randomInt(hw)+Calc.randomInt(hw))/2 + minh;
            generate(count);
        } while( !isPossiblePattern() );

        return pattern;
    }

    // ヒント数を指定してヒントパターンを作成
    public boolean[][] generate(int hintcount) {
        clear();
        int count = 0;

        if( (hintcount % 2) == 1) { // 奇数個のとき中央に置く
            pattern[4][4] = true;
            ++count;
        }
        
        boolean foursymmetric  = Calc.randomInt(8) < 7;
        boolean eightsymmetric = Calc.randomInt(8) < 7;
        
        while( count < hintcount ) {
            int x = Calc.randomInt(9);
            int y = Calc.randomInt(9);
            if( pattern[x][y] || (x==4 && y==4) )
                continue;
            int z = Calc.zone(x,y);
            if( countHintsZ(z) > (z==4?3:5) 
                    || countHintsX(x) > (x==4?4:5) 
                    || countHintsY(y) > (y==4?4:5) )
                continue;
            
            pattern[x][y] = true;
            ++count;
            pattern[8-x][8-y] = true;
            ++count;
            if( foursymmetric && (hintcount - count)>=2) {
                pattern[y][8-x] = true;
                ++count;
                pattern[8-y][x] = true;
                ++count;
            }
            if( eightsymmetric && (hintcount - count)>=4 ) {
                if( ! pattern[8-x][y] ) {
                    pattern[8-x][y] = true;
                    ++count;
                }
                if( ! pattern[x][8-y] ) {
                    pattern[x][8-y] = true;
                    ++count;
                }
                if( ! pattern[8-y][8-x] ) {
                    pattern[8-y][8-x] = true;
                    ++count;
                }
                if( ! pattern[y][x] ) {
                    pattern[y][x] = true;
                    ++count;
                }           
            }
        }
    
        return pattern;
    }
    
    private void clear() {
        for( int x=0; x<9; ++x )
            for( int y=0; y<9; ++y )
                pattern[x][y] = false;
    }

    public int getHintCount() {
        int n=0;
        for( int x=0; x<9; ++x )
            for( int y=0; y<9; ++y )
                if( pattern[x][y] )
                    ++n;
        return n;
    }
    
    public void printPattern(boolean [][] pat) {
        System.out.println("pattern");
        for( int y=0; y<9; ++y ) {
            for( int x=0; x<9; ++x) {
                System.out.print( pat[x][y] ? "X " : ". " );
            }
            System.out.println();
        }
    }

    /** パターンチェック（明らかに不可能なものを除く）   */  
    public boolean isPossiblePattern() {
        
        int[] xh = new int[9];
        int[] yh = new int[9];
        int[] zh = new int[9];
        for( int i=0; i<9; ++i ) {
            xh[i] = countHintsX(i);
            yh[i] = countHintsY(i);
            zh[i] = countHintsZ(i);
        }

        // 空白縦列、横列チェック
        for( int i=0; i<9; i+=3 ) {
            int xzerocount = 0;
            int yzerocount = 0;
            for( int j=0; j<3; ++j ) {
                if( xh[i+j] == 0 )  ++xzerocount;
                if( yh[i+j] == 0 )  ++yzerocount;
            }
            if( xzerocount > 1 || yzerocount > 1 )
                return false;
        }
        
        int zhzerocount = 0;
        int[] zhxz = new int[3];
        int[] zhyz = new int[3];
        for( int i=0; i<9; ++i )
            if( zh[i] == 0 ) {
                ++ zhzerocount;
                ++ zhxz[i/3];
                ++ zhyz[i%3];
            }
        
        if( zhzerocount >= 3 ) {
            for( int i=0; i<9; ++i )
                if( zh[i] == 0 ) {
                    if( zhxz[i/3] == 2 && zhyz[i%3] == 2 )
                        return false;
                }
        }

        if( zhzerocount >= 2 ) {
            for( int i=0; i<3; ++i ) {
                if( (zhxz[i] == 2) && (zh[i*3]+zh[i*3+1]+zh[i*3+2] < 4) )
                    return false;
                if( (zhyz[i] == 2) && (zh[i%3]+zh[i%3+3]+zh[i%3+6] < 4) )
                    return false;
            }
        }
        
        return true;
    }

    private int countHintsX( int i ) {
        int n = 0;
        for( int j=0; j<9; ++j )
            n += pattern[i][j] ? 1 : 0;
        return n;
    }
    private int countHintsY( int j ) {
        int n = 0;
        for( int i=0; i<9; ++i )
            n += pattern[i][j] ? 1 : 0;
        return n;
    }
    private int countHintsZ( int i ) {
        int n = 0;
        for( int j=0; j<9; ++j ) {
            int x = Calc.ztox(i, j);
            int y = Calc.ztoy(i, j);
            n += pattern[x][y] ? 1 : 0;
        }
        return n;
    }
} 
